//
//  TodoList.swift
//  Do It
//
//  Created by Jim Dovey on 8/25/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI

struct TodoList: View {
    // START:NewListData
    private enum ListData {
        case list(TodoItemList)
        case items(LocalizedStringKey, [TodoItem])
    }
    
    // END:NewListData
    @State private var sortBy: SortOption = .title
    @State private var showingChooser: Bool = false
    @State private var showingListEditor: Bool = false
    @Environment(\.presentationMode) private var presentationMode
    @EnvironmentObject private var data: DataCenter

    // START:NewListData
    @State private var listData: ListData
    // END:NewListData

    init(list: TodoItemList) {
        self._listData = State(wrappedValue: .list(list))
    }

    init(title: LocalizedStringKey, items: [TodoItem]) {
        self._listData = State(wrappedValue: .items(title, items))
    }

    var body: some View {
        List {
            ForEach(sortedItems) { item in
                NavigationLink(destination: TodoItemDetail(item: item)
                    .environmentObject(self.data)
                ) {
                    TodoItemRow(item: item)
                        .accentColor(self.color(for: item))
                }
            }
        }
        .navigationBarTitle(title)
        .navigationBarItems(trailing: barItems)
        .listStyle(GroupedListStyle())
        .actionSheet(isPresented: $showingChooser) {
            ActionSheet(
                title: Text("Sort Order"),
                buttons: SortOption.allCases.map { opt in
                    ActionSheet.Button.default(Text(opt.title)) {
                        self.sortBy = opt
                    }
            })
        }
        .sheet(isPresented: $showingListEditor) {
            TodoListEditor(list: self.list!)
                .environmentObject(self.data)
        }
    }
}

// MARK: - Helper Properties

extension TodoList {
    private var sortButton: some View {
        Button(action: { self.showingChooser.toggle() }) {
            Image(systemName: "arrow.up.arrow.down.square")
                .imageScale(.large)
                .font(.system(size: 24, weight: .bold))
        }
    }

    private var barItems: some View {
        HStack(spacing: 14) {
            if isList {
                Button(action: { self.showingListEditor.toggle() }) {
                    Image(systemName: "info.circle")
                        .imageScale(.large)
                        .font(.system(size: 24, weight: .bold))
                }
            }
            sortButton
        }
    }

    private var isList: Bool {
        if case .list = self.listData {
            return true
        }
        return false
    }
    
    private var list: TodoItemList? {
        if case let .list(list) = self.listData {
            return list
        }
        return nil
    }

    private func forciblyDismiss() {
        presentationMode.wrappedValue.dismiss()
    }

    private var items: [TodoItem] {
        switch listData {
        case .list(let list): return data.items(in: list)
        case .items(_, let items): return items
        }
    }

    private var title: LocalizedStringKey {
        switch listData {
        case .list(let list): return LocalizedStringKey(list.name)
        case .items(let name, _): return name
        }
    }

    private func color(for item: TodoItem) -> Color {
        switch listData {
        case .list(let list): return list.color.uiColor
        case .items: return data.list(for: item).color.uiColor
        }
    }
}

// MARK: - Sorting

extension TodoList {
    private var sortedItems: [TodoItem] {
        return items.sorted {
            switch sortBy {
            case .title:
                return $0.title.lowercased() < $1.title.lowercased()
            case .priority:
                return $0.priority > $1.priority
            case .dueDate:
                return ($0.date ?? .distantFuture) <
                    ($1.date ?? .distantFuture)
            }
        }
    }
}

// MARK: - Model Manipulation

extension TodoList {
    
}

fileprivate enum SortOption: String, CaseIterable {
    case title = "Title"
    case priority = "Priority"
    case dueDate = "Due Date"

    var title: LocalizedStringKey { LocalizedStringKey(rawValue) }
}

struct TodoList_Previews: PreviewProvider {
    static var previews: some View {
        let data = DataCenter()
        return NavigationView {
            TodoList(title: "All Items", items: data.todoItems)
                .environmentObject(DataCenter())
        }
    }
}
